define([
    'backbone',
    'marionette',
    'App',
    'Vent',
    'User',
    'modules/connection/connection-error-popup-view',
    'modules/connection/server-error-popup-view',
    'modules/connection/session-ended-popup-view',
    'modules/connection/session-warning-popup-view',
],
function (
    Backbone,
    Marionette,
    app,
    vent,
    User,
    ConnectionErrorPopupView,
    ServerErrorPopupView,
    SessionEndedPopupView,
    SessionWarningPopupView
) {
    'use strict';
    var utilities = {
        cleanUpSession: function () {
            sessionStorage.clear();
            $.ajaxPrefilter(function (options, originalOptions, jqXHR) {
                jqXHR.setRequestHeader('Authorization', '');
            });
        },
    };

    var connectionModule = {
        models: {
            user: User,
            userSession: app.userSession,
        },
        serverErrorsQueue: new Backbone.Collection(),
        showServerError: function (jqxhr, settings, messageType, errorMessage, errorType) {
            var serverErrorModel;

            this.closeModal('.ui-popup-active .ui-popup:not(.server-error)');
            serverErrorModel = new Backbone.Model({
                jqxhr: jqxhr,
                settings: settings,
                messageType: messageType ? messageType : 'default',
                errorMessage: errorMessage ? errorMessage : '',
                errorType: errorType ? errorType : 'Error',
            });
            this.serverErrorsQueue.add(serverErrorModel);

            $.when(app.showingContent).then(function() {
                if ($('.server-error').length === 0) {
                    this._showServerError(this.serverErrorsQueue.first());
                }
            }.bind(this));
        },
        _showServerError: function(model) {
            var popup = new ServerErrorPopupView({
                elAfterClose: $('.primary-header h2'),
                model: model,
            });
            popup.once('before:destroy', function() {
                this.serverErrorsQueue.remove(model);
            }, this);

            popup.openPopup();
        },
        showSessionWarning: function () {
            this.closeModal();

            if ($('#session-warning').length === 0) {
                (new SessionWarningPopupView({
                    model: app.userSession,
                    elAfterClose: $('.primary-header h2'),
                })).openPopup();
            }
        },
        showSessionExpired: function () {
            this.closeModal();
            this.models.user.clear();

            (new SessionEndedPopupView({elAfterClose: $('.primary-header h2')})).openPopup();

            utilities.cleanUpSession();
        },
        showConnectionError: function () {
            this.closeModal();

            (new ConnectionErrorPopupView({elAfterClose: $('.primary-header h2')})).openPopup();
        },
        closeConnectionError: function () {
            this.closeModal('#connection-error');
        },
        closeModal: function(selector) {
            $(selector || '.ui-popup-active .ui-popup').popup('close');
        },
        sessionRefresh: function() {
            // any call to the api gateway will refresh the token expiration
            $.ajax({url: app.resources.get('jwt-refresh').get('href')});
        },
    };

    $(document).on('popupafterclose', '.server-error', function() {
        if (connectionModule.serverErrorsQueue.length) {
            connectionModule._showServerError(connectionModule.serverErrorsQueue.first());
        }
    });

    vent.on('show:session:warning', function () {
        connectionModule.showSessionWarning();
    });

    vent.on('show:session:expired', function () {
        connectionModule.showSessionExpired();
    });

    vent.on('show:server:error', function (jqxhr, settings, messageType, errorMessage, errorType) {
        connectionModule.showServerError(jqxhr, settings, messageType, errorMessage, errorType);
    });

    vent.on('show:connection:error', function () {
        connectionModule.showConnectionError();
    });
    vent.on('close:connection:error', function () {
        connectionModule.closeConnectionError();
    });

    vent.on('do:session:refresh', function() {
        connectionModule.sessionRefresh();
    });

    return connectionModule;
});
